# Function VarSel() is used to select important variables in a 
# reproducing kernel. This is based on the Metropolis-Hastings 
# algorithm.
# Input:
# 	Y: response variable (nx1)
# 	X: matrix of covariates (nxnumCov)
#	Candid: matrix of candidate variables (nxp)
# 	numCov: number of covariates
# 	numItr: number of iterations per chain
# 	a0 and b0 are for the prior of sigma^2, sig2. We use the same for tau
# 	sigB2 is the variance in the prior of beta

# Output: this includes three pieces
# 	a)  quantiles obtained by treating (n - n1) x numChain matrix as samples  
#         from posterior of the first beta and tau, where n1 is the number of iterations after burn-in.
#     b) Gelman and Rubin's convergence measure R and an upper confidence
#         limit for R (values near 1 are desirable) for each of beta and tau
#	c) model size
#	d) indices of selected variables
# 	e) posterior probability of selecting each variable
# 	f) ordered posterior probabilities of each variable

VarSel<-function(Y, X, Candid, rho=1, numCov=1, numItr=50000, numChain=1, a0=0.001,b0=0.001,sigB2=10)
{
	sampsize<-length(Y)
	numCandid<-ncol(Candid) 
	# Gaussian kernel is used
	distance<-array(0,dim=c(numCandid,sampsize,sampsize))

	for (i in 1:(sampsize))
	{
		distance[,,i]<-t((Candid - matrix(1, ncol=1, nrow=sampsize)%*%(as.matrix(Candid[i,])))^2)
	}
	paramArray<-array(0,dim=c(numItr,numChain,numCov+1))

	# for the proposal distribution of tau
	UppTau<-2000
	stdTau<-1

	# for the proposal distribution of sigma^2
	stdsigma<-0.7

	delta<-rbinom(numCandid,1,0.5)
	K<-updateK(delta,distance,sampsize,rho)
	for (i in 1:numChain)
	{
		set.seed(seed=1000*i)

		beta<-rnorm(numCov)
		sig2<-0.1*i 
		tau<-i
		h<-as.vector(rmnorm(n=1,0,tau*K))
		q<-rep(0.5,numCandid)

		for (j in 1:numItr)
		{
			#update beta
			SolveSigma<-solve(tau*K+diag(sig2,sampsize))
			SigmaBeta<-solve(t(X)%*%SolveSigma%*%X+diag((sigB2)^(-1),numCov))
			muBeta<-SigmaBeta%*%t(X)%*%SolveSigma%*%Y
			beta<-rmnorm(n=1,muBeta,SigmaBeta)

			# update tau
		
			eigVal<-eigen(SolveSigma,symmetric=TRUE,only.values=TRUE)$values
			detSolveSigma<-min(max(prod(eigVal),10^(-322)),10^300)
			lhOld<-(-0.5)*t(Y-X%*%beta)%*%SolveSigma%*%(Y-X%*%beta)+0.5*log(detSolveSigma)+(-a0-1)*log(tau)-b0/tau	

			TauProp<-exp(rnorm(1,log(tau),stdTau))
		
			while (TauProp>UppTau)
			{
				TauProp<-exp(rnorm(1,log(tau),stdTau))
			}
			SigTemp<-TauProp*K+diag(1,ncol(K))*sig2
			SolvesigTmp<-solve(SigTemp)
			eigVal<-eigen(SolvesigTmp,symmetric=TRUE,only.values=TRUE)$values
			detSolvesigTmp<-min(max(prod(eigVal),10^(-322)),10^300)

			lhNew<-(-0.5)*t(Y-X%*%beta)%*%SolvesigTmp%*%(Y-X%*%beta)+0.5*log(detSolvesigTmp)+(-a0-1)*log(TauProp)-b0/TauProp
			AccRatio<-lhNew-lhOld+log(TauProp)-log(tau)	
			judge<-runif(1)
			if (AccRatio>log(judge))			
			{
				tau<-TauProp
				SolveSigma<-SolvesigTmp
				detSolveSigma<-detSolvesigTmp
			}

			# update sigma^2, sig2
			lhOld<-(-0.5)*t(Y-X%*%beta)%*%SolveSigma%*%(Y-X%*%beta)+0.5*log(detSolveSigma)+(-a0-1)*log(sig2)-b0/sig2
			sigProp<-exp(rnorm(1,log(sig2),stdsigma))
			SigTemp<-tau*K+diag(1,ncol(K))*sigProp
			SolvesigTmp<-solve(SigTemp)
			eigVal<-eigen(SolvesigTmp,symmetric=TRUE,only.values=TRUE)$values
			detSolvesigTmp<-min(max(prod(eigVal),10^(-322)),10^300)

			lhNew<-(-0.5)*t(Y-X%*%beta)%*%SolvesigTmp%*%(Y-X%*%beta)+0.5*log(detSolvesigTmp)+(-a0-1)*log(sigProp)-b0/sigProp
			AccRatio<-lhNew-lhOld+log(sigProp)-log(sig2)	
			judge<-runif(1)
			if (AccRatio>log(judge))			
			{
				sig2<-sigProp
				SolveSigma<-SolvesigTmp
				detSolveSigma<-detSolvesigTmp
			}

			paramArray[j,i,1:numCov]<-beta
			paramArray[j,i,numCov+1]<-tau
			
			# update delta
	
			randIndex<-sample(numCandid,numCandid)
			for (kk in 1:numCandid)
			{
				s<-randIndex[kk]
				a<-(-0.5)*t(Y-X%*%beta)%*%SolveSigma%*%(Y-X%*%beta)+0.5*log(detSolveSigma)

				deltaTmp<-delta
				deltaTmp[s]<-1-delta[s]
				newK<-updateK(deltaTmp,distance,sampsize,rho)
				SigTemp<-tau*newK+diag(1,ncol(K))*sig2
				SolvesigTmp<-solve(SigTemp)
				eigVal<-eigen(SolvesigTmp,symmetric=TRUE,only.values=TRUE)$values
				detSolvesigTmp<-min(max(prod(eigVal),10^(-322)),10^300)

				b<-(-0.5)*t(Y-X%*%beta)%*%SolvesigTmp%*%(Y-X%*%beta)+0.5*log(detSolvesigTmp)

				maxab<-max(a,b)
				prob<-exp(a*delta[s]+b*(1-delta[s])-maxab)/(exp(a-maxab)+exp(b-maxab))
				deltaTmp[s]<-rbinom(1,1,prob)
	
				if (deltaTmp[s]!=delta[s])
				{
					delta[s]<-deltaTmp[s]
					K<-updateK(delta,distance,sampsize,rho)
					SolveSigma<-SolvesigTmp
					detSolveSigma<-detSolvesigTmp
				}
#				q[s]<-rbeta(1,aa+delta[s],bb+1-delta[s])
				q[s]<-0.5
			} # end for (kk in 1:numCandid)
			if (j==ceiling(numItr*2/3+1) && i==1)
			{
				sumDelta<-rep(0,length(delta))
				kkk<-0
			}
			else if (j>ceiling(numItr*2/3+1) && i==1)
			{
				sumDelta<-sumDelta+delta
				kkk<-kkk+1
			}
		}#end numItr
		
		sumDelta1<-sumDelta/kkk
	
		propDelta<-cbind(seq(1,numCandid),sumDelta1)
		colnames(propDelta)<-c("Variable Index","Post. Prob")
		orderProp<-propDelta[order(propDelta[,2],decreasing=T),]
		maxDiff<-max(abs(diff(orderProp[,2])))
		VarIndex<-which(abs(diff(orderProp[,2]))==maxDiff)
		VarSelect<-orderProp[1:VarIndex,1]
		VarSelect<-VarSelect[order(VarSelect)]
		
		modelSize<-length(VarSelect)
	}#end chain

	diagnosis1<-gandr.conv(paramArray[,,1])
	diagnosis2<-gandr.conv(paramArray[,,numCov+1])
	list(diagB1=diagnosis1,diagTau=diagnosis2,
		modelSize=modelSize,propDelta=propDelta,orderProp=orderProp,VarSelect=VarSelect)
}
VarSelResults<-VarSel(Y, X, Candid, rho=1, numCov=1, numItr=50, numChain=2, a0=0.001,b0=0.001,sigB2=10)